# scripts/run_path_integral.py
#!/usr/bin/env python3
import argparse
import json
import os

from ar_sim.common.fractal_fits import load_D_values
from ar_sim.path_integral import PathIntegralSampler


def main():
    parser = argparse.ArgumentParser(
        description="Run the Continuum Hamiltonian → Path Integral pipeline"
    )
    parser.add_argument(
        "--sweep", type=int, default=100000,
        help="Number of Metropolis–Hastings sweeps"
    )
    parser.add_argument(
        "--step-size", type=float, default=0.1,
        help="Proposal step size for field updates"
    )
    parser.add_argument(
        "--temp", type=float, default=1.0,
        help="Effective temperature (1/kT) scaling for Metropolis acceptance"
    )
    parser.add_argument(
        "--sigma", type=float, default=1.0,
        help="Kernel width parameter for continuum action"
    )
    parser.add_argument(
        "--save-interval", type=int, default=1000,
        help="Interval (in sweeps) at which to record samples"
    )
    parser.add_argument(
        "--output", type=str, default="results/samples.h5",
        help="Path to HDF5 file for saving samples"
    )
    parser.add_argument(
        "--pivot-params", type=str, default="data/pivot_params.json",
        help="JSON file with pivot parameters {a, b}"
    )
    args = parser.parse_args()

    # Load fractal-dimension anchors
    n_vals, D_vals, sigma_vals = load_D_values()

    # Load pivot parameters
    with open(args.pivot_params, 'r') as f:
        pivot = json.load(f)
    pivot_params = {
        'a': pivot['a'],
        'b': pivot['b'],
        'D_vals': D_vals
    }

    # Ensure output directory exists
    out_dir = os.path.dirname(args.output)
    if out_dir and not os.path.exists(out_dir):
        os.makedirs(out_dir)

    # Initialize and run sampler
    sampler = PathIntegralSampler(
        n_vals=n_vals,
        pivot_params=pivot_params,
        sigma=args.sigma,
        sweep=args.sweep,
        step_size=args.step_size,
        temperature=args.temp,
        save_interval=args.save_interval,
        output=args.output
    )
    sampler.run()


if __name__ == "__main__":
    main()
